/*************************************************************************************************
 *
 *   Copyright (c) Hilscher GmbH. All Rights Reserved.
 *
 **************************************************************************************************
 * @file Application.cpp
 *
 * @brief This file contains some common functions which might be used by any example application. 
 * 
 * @author Elmar
 *
 */

#include "stdafx.h"
#include "Debug.h"
#include "Common.h"
#include "Application.h"


/*************************************************************************************************
 * @brief This function sends a packet and waits for the confirmation.
 * Packets which will come up meanwhile will be passed to the standard packet handler.
 * 
 * @param ptApp     Pointer to application data.
 * @param ptSendPkt Packet which will be send.
 * @param ptRecvPkt Packet which the function has received.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_SendRecvPkt(APPLICATION_T* ptApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
  TLR_RESULT tResult = TLR_S_OK; 

  /* fire the packet */
  tResult = xChannelPutPacket(ptApp->tCommon.hChannel, ptSendPkt, ptApp->tCommon.ulTimeout);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* ok, at this point we have successfully sent a packet */

  /* check for packets to receive */
  /* the ulCnfCmd is always: ulReqCmd | 0x01 */
  TLR_UINT32 ulCnfCmd = ptSendPkt->tHeader.ulCmd | 0x01;  
  
  while(CIFX_NO_ERROR == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(*ptRecvPkt), ptRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    /* check for our packet */
    if(ptRecvPkt->tHeader.ulCmd == ulCnfCmd)
    {
      /* it is our packet, so return its status as result */
      tResult = ptRecvPkt->tHeader.ulState;
      
      /* Note: we also return the packet which we have received (by reference, see signature of function) */

      /* we have received our packet, so we can break here */
      break;
    }
    else
    {
      /* it is something else, so place it in the application packet handler */
      App_HandlePacket(ptApp, ptRecvPkt);
    }
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This function sends and receives an empty packet.
 * 
 * @param ptApp Pointer to application data.
 * @param ulCmd Command code which will be inserted into packet header.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_SendRecvEmptyPkt(APPLICATION_T* ptApp, TLR_UINT32 ulCmd)
{
  
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  TLR_RESULT tResult = TLR_S_OK;

  /* all register packets are empty packet, */
  /* which means that they contain no data, */
  /* so a cast to a specific type is not necessary */
  TLR_EMPTY_PACKET_T* ptReq = (TLR_EMPTY_PACKET_T*) &tSendPkt;

  /* ulSrc and ulDest are set by GetPacket / SendPacket */
  ptReq->tHead.ulCmd    = ulCmd;
  ptReq->tHead.ulDest   = 0x20;
  ptReq->tHead.ulDestId = 0;
  ptReq->tHead.ulExt    = 0;
  ptReq->tHead.ulId     = 0;
  ptReq->tHead.ulLen    = 0;
  ptReq->tHead.ulRout   = 0;
  ptReq->tHead.ulSrc    = 0;
  ptReq->tHead.ulSrcId  = 0;
  ptReq->tHead.ulSta    = 0;

  tResult = App_SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  return tResult;
}



/*************************************************************************************************
 * @brief This is the main entry function for the application
 * 
 * @param ptApp Pointer to application data.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT App_Main(APPLICATION_T* ptApp)
{
  /* return value of main function */
  TLR_RESULT tResult = TLR_S_OK;
  
  /* name of board to use */
  char* szBoardName = "CIFx0";

  /* number of channel to use */
  TLR_UINT32 ulChannel = 0;

  /* general reference to driver */
  CIFXHANDLE  hDriver  = NULL;

  /* general reference to communication channel */
  CIFXHANDLE  hChannel = NULL;

  /* common variables for packets */
  CIFX_PACKET tSendPkt  = {{0}};
  CIFX_PACKET tRecvPkt  = {{0}};
  CIFX_PACKET* ptRecvPkt = NULL;

  /* variable for host state */
  TLR_UINT32  ulState     = 0;

  /* default timeout for driver api calls */
  TLR_UINT32  ulTimeout   = 100;
  /* which key is pressed? */
  TLR_INT iKey = 0;

  /* prompt a startup screen when running in debug mode */
  tResult = App_PromptIntro(ptApp);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* place default timeout value in resources */
  ptApp->tCommon.ulTimeout = ulTimeout;

  /* Open Driver */
  DEBUG("Opening driver...\n");
  tResult = xDriverOpen(&hDriver);
  
  if (CIFX_NO_ERROR == tResult)
  {
    /* place driver handle in application resources */
    ptApp->tCommon.hDriver = hDriver;

    /* Driver successfully opened */
    /* Open channel */
    DEBUG("Opening channel %d on board %s...\n", ulChannel, szBoardName);
    tResult = xChannelOpen(hDriver, szBoardName, ulChannel, &hChannel);

    if (CIFX_NO_ERROR == tResult)
    {
      /* place channel handle in application resources */
      ptApp->tCommon.hChannel = hChannel;

      /* start hardware with new configuration */
      DEBUG("Processing system restart...\n");

#ifdef WIN32
      tResult = xChannelReset(hChannel, CIFX_SYSTEMSTART, 2000);
#endif
      
      /* system restart successful */
      if (CIFX_NO_ERROR == tResult)
      {
        /* Toggle Application Ready State Flag */
        do
        {
          tResult = xChannelHostState(hChannel, CIFX_HOST_STATE_READY, &ulState, ulTimeout);
          
          /* if Dev is not ready, retry that action */
          if (CIFX_DEV_NOT_READY == tResult)
          {
            /* retry after 500 ms */
            Sleep(500);
          }
        }
        while (tResult == CIFX_DEV_NOT_READY);      

		/* initialize the application */
        tResult = App_Initialize(ptApp);


        /* check for CIFX_NO_ERROR added because return value  of xChannelHostState() changed */
        /* between SHM-lib <=0.930 and >=0.940 */
        if ((CIFX_NO_ERROR == tResult) || (CIFX_DEV_NOT_RUNNING == tResult))
        {
			DEBUG("Assuming configuration to exist in EtherCAT Master.\n");

          {
            if (CIFX_NO_ERROR == tResult)
            {
              /* Waiting for netX to use configuration */
              do
              {
                tResult = xChannelHostState(hChannel, CIFX_HOST_STATE_READY, &ulState, ulTimeout);
              }
              while (CIFX_DEV_NOT_RUNNING == tResult);
            
            
              /* check CifX state */
              if ((CIFX_NO_ERROR == tResult) || (CIFX_DEV_NO_COM_FLAG == tResult))
              {
                {
                  /* bus on */
                  DEBUG("Setting bus state on and wait for communication to be established...\n");
                  tResult = xChannelBusState(hChannel, CIFX_BUS_STATE_ON, &ulState, 2000);        
          
                  /* bus activity begins here */
                  if (CIFX_BUS_STATE_ON == ulState)
                  {
                    DEBUG("Entering endless loop...\n");
                    DEBUG("\n");

					/* get and show handles of configured devices */
					App_GetHandles(ptApp);

					/* print the help to see what application can do */
                    App_ShowHelp();
          
                    /* set return to TLR_S_OK to initially enter the loop */
                    tResult = TLR_S_OK;
                    while ((TLR_S_OK == tResult || CIFX_DEV_GET_NO_PACKET == tResult) && iKey != 113) /* 113 is 'q' */
                    {
                      /* do process stuff first */
                      /* handling of io depends on the bus system,  */
                      /* there might be some bits we have to toggle */
					  App_HandleProcessData(ptApp);
            
                      /* now we do the acyclic stuff */
                      tResult = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, 5);
            
                      /* check if we have really received a packet or an error code */
                      if(TLR_S_OK == tResult)
                      {
                        /* handle packets */
                        App_HandlePacket(ptApp, &tRecvPkt);
                      }

					  /* check (and print) when amount of slaves in certain state has changed */
					  App_GetSlaveState(ptApp, &tSendPkt, &tRecvPkt);

#ifdef WIN32
                      /* enable leaving of endless loop if running under windows, */
                      /* necessary to return handles and close driver */
                      if (_kbhit())
                      {
                        iKey = _getch();
						App_HandleKey(ptApp, iKey);
                      }
#endif
                          
                    } /* while(TLR_S_OK == tResult || CIFX_DEV_GET_NO_PACKET == tResult) --> packet loop */
  
                    /* force at least one empty line within console output */
                    DEBUG("\n\n");
  
                  } /* Bus on */
				  else
				  {
					  DEBUG("bus on FAILED: ulState is 0x%08x\n", ulState);
				  }

                  /* Finalize application in order to dispose handles, free memory, aso.*/
                  App_Finalize(ptApp);
  
                } /* check initialize application succeeded */

              } /* check CifX state */

            } /* cifX channel init */

          } /* check configuration packet */

        } /* check for CIFX_NO_ERROR added because return value of xChannelHostState() changed */
		else
		{
			DEBUG("error: 0x%08x\n", tResult);
		}

      } /* cifX successfully reset */
	  else
	  {
		DEBUG("SystemReset returned error: 0x%08x\n", tResult);
	  }

      
      DEBUG("Closing Channel...\n");
      tResult = xChannelClose(hChannel);
    } /* channel successfully opened */
	else
	{
	  DEBUG("open Channel returned 0x%08x\n", tResult);
	}

    DEBUG("Closing Driver...\n");
    tResult = xDriverClose(hDriver);

  } /* driver successfully opened */

  DEBUG("leave application with error 0x%08x in 3 seconds.\n", tResult);
  Sleep(3000);
  
  /* bye, bye, ...*/
  return tResult;
}


void App_ShowHelp()
{
	DEBUG("\n");
	DEBUG("key        Action\n");
	DEBUG("h          show this help\n");
	DEBUG("g          get DeviceHandles and print out some information\n");
	DEBUG("u          perform acyclic CoE Upload (requires slave mailbox support)\n");
	DEBUG("q          quit application\n");
	DEBUG("\n");
}




